package de.fhw.swtvertiefung.knauf.security.loginmodule;

import java.security.acl.Group;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;

import javax.naming.InitialContext;
import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.LoginException;
import javax.sql.DataSource;

import org.jboss.security.SimpleGroup;
import org.jboss.security.SimplePrincipal;
import org.jboss.security.auth.spi.UsernamePasswordLoginModule;

/**Login-Modul fr den Fall Kunde/Administrator. 
 * 
 * @author Wolfgang Knauf
 *
 */
public class KundeAdministratorLoginModule extends UsernamePasswordLoginModule
{
  /**JNDI-Name der Datasource. */
  private String DATASOURCE_JNDI = null;

  /**Datenbankschema. */
  private String DBSCHEMA = null;

  /**Die Datenquelle auf der wir arbeiten. Liefert uns die eigentlichen Connections.*/
  private DataSource dataSource = null;

  /** Die Rolle "Administrator". */
  private final String roleAdmin = "administrator";

  /** Die Rolle "Kunde". */
  private final String roleKunde = "kunde";

  /**Initialisierung des Login-Moduls (wird einmalig bei Initialisierung des Moduls aufgerufen):
   * hier werden die Config-Parameter "DataSource-JNDI-Name" und "DB-Schema" ausgelesen.
   * @param subject
   * @param callbackHandler
   * @param sharedState
   * @param options ConfigParameter, hier werden "DATASOURCE_JNDI" und "DBSCHEMA" erwartet.
   */
  public void initialize(Subject subject, CallbackHandler callbackHandler, Map sharedState, Map options)
  {
    super.initialize(subject, callbackHandler, sharedState, options);
    
    try
    {
      //Config-Parameter auslesen:
      this.DATASOURCE_JNDI = (String) options.get("DATASOURCE_JNDI");
      this.DBSCHEMA = (String) options.get("DBSCHEMA");

      //Schemaangabe NULL in Leerstring umwandeln, sonst pat String-Concatenation nicht.
      if (this.DBSCHEMA == null)
        this.DBSCHEMA = "";

      InitialContext ctx = new InitialContext();
      //DataSource ds = (DataSource) ctx.lookup("java:/DefaultDS");
      this.dataSource = (DataSource) ctx.lookup(this.DATASOURCE_JNDI);
      //Connection wird nur bei Bedarf auf- und wieder zugemacht.
      log.error("initialize: Datenquelle geholt");
    }
    catch (Exception e)
    {
      log.error("initialize: Datenquelle konnte nicht gefunden werden");
      log.error(e);
    }

  }

  /**Die Rollen des aktuellen Users ermitteln. 
   * @return Ein Array von Gruppen des Users, entweder "administrator" oder "kunde".
   * @throws LoginException Falls User nicht angemeldet werden konnte.
   */
  protected Group[] getRoleSets() throws LoginException
  {
    //Ist das ein Kunde oder ein Admin ??
    try
    {
      //Die Gruppen initialisieren. 
      Group[] groups = { new SimpleGroup("Roles") };

      //Kunde laden ?
      Kunde kunde = this.loadKundeByLogin(super.getUsername());
      if (kunde != null)
      {
        //Kunde mit Login wurde gefunden:
        log.error("getRoleSets for Login '" + super.getUsername() + "' -> Kunde gefunden");
        SimplePrincipal role = new SimplePrincipal(roleKunde);
        groups[0].addMember(role);
        return groups;
      }
      else
      {
        //Ist es ein Admin ?
        Administrator admin = this.loadAdminByLogin(super.getUsername());
        if (admin != null)
        {
          //Admin mit Login wurde gefunden:
          log.error("getRoleSets for Login '" + super.getUsername() + "' -> Admin gefunden");
          SimplePrincipal role = new SimplePrincipal(roleAdmin);
          groups[0].addMember(role);
          return groups;
        }
        else
        {
          //Weder Admin noch Kunde fr Login gefunden.
          throw new LoginException("Weder Kunde noch Admin fr ID '" + super.getUsername() + "' gefunden.");
        }
      }
    }
    catch (SQLException e)
    {
      log.error("Fehler in getUsersPassword fr User " + super.getUsername(), e);
      throw new LoginException(e.getMessage());
    }
  }

  /**Passwort des aktuellen Users (der in der Basisklasse bereits vom Server gesetzt wurde)
   * ermitteln.
   * @return Passwort des Users.
   * @throws LoginException Falls User nicht gefunden wurde.
   */
  protected String getUsersPassword() throws LoginException
  {
    //Ist das ein Kunde oder ein Admin ??
    try
    {
      //Kunde laden ?
      Kunde kunde = this.loadKundeByLogin(super.getUsername());
      if (kunde != null)
      {
        //Kunde mit Login wurde gefunden:
        log.error("getUsersPassword for Login '" + super.getUsername() + "' -> Kunde gefunden");
        return kunde.getPasswort();
      }
      else
      {
        //Ist es ein Admin ?
        Administrator admin = this.loadAdminByLogin(super.getUsername());
        if (admin != null)
        {
          //Admin mit Login wurde gefunden:
          log.error("getUsersPassword for Login '" + super.getUsername() + "' -> Admin gefunden");
          return admin.getPasswort();
        }
        else
        {
          //Weder Admin noch Kunde fr Login gefunden.
          log.error("getUsersPassword for Login '" + super.getUsername() + "' -> weder Kunde noch Admin gefunden");
          throw new LoginException("Weder Kunde noch Admin fr ID '" + super.getUsername() + "' gefunden.");
        }
      }
    }
    catch (SQLException e)
    {
      log.error("Fehler in getUsersPassword fr User " + super.getUsername(), e);
      throw new LoginException(e.getMessage());
    }
  }

  /**
   * Laden eines Kunden anhand des Logins.
   * 
   * @return Der Kunde oder NULL, wenn nicht gefunden.
   */
  private Kunde loadKundeByLogin(String str_LoginName) throws SQLException
  {
    Connection con = this.dataSource.getConnection();
    Statement stmt = con.createStatement();
    
    String strSchema = "";
    if (this.DBSCHEMA != null)
      strSchema = this.DBSCHEMA + ".";
    ResultSet rs = stmt.executeQuery("SELECT ID, NAME, LOGIN, PASSWORT FROM " + strSchema + "KUNDE WHERE LOGIN='"
        + str_LoginName + "'");
    Kunde kunde = null;
    if (rs.next())
    {
      kunde = new Kunde();
      kunde.fromResultSet(rs);
    }
    //Allgemeines aufrumen:
    rs.close();
    stmt.close();
    //Connection hat Autocommit gesetzt => kein Rollback/Commit ntig
    con.close();
    
    return kunde;
  }

  /**
   * Laden eines Administrators anhand des Logins.
   * 
   * @return Den Administrator oder NULL, wenn nicht gefunden.
   */
  private Administrator loadAdminByLogin(String str_LoginName) throws SQLException
  {
    Connection con = this.dataSource.getConnection();
    Statement stmt = con.createStatement();
    
    String strSchema = "";
    if (this.DBSCHEMA != null)
      strSchema = this.DBSCHEMA + ".";
    ResultSet rs = stmt.executeQuery("SELECT ID, NAME, LOGIN, PASSWORT FROM " + strSchema
        + "ADMINISTRATOR WHERE LOGIN='" + str_LoginName + "'");
    Administrator admin = null;
    if (rs.next())
    {
      admin = new Administrator();
      admin.fromResultSet(rs);
    }
    //Allgemeines aufrumen:
    rs.close();
    stmt.close();
    //Connection hat Autocommit gesetzt => kein Rollback/Commit ntig
    con.close();
    return admin;
  }

}
